/* 
 Name:
	Create3DMengerSponge
 Version:
	2.6 (15 August 2010)
 Author:
	Charles Bordenave
 Description:  
	This script creates a 3D Menger Sponge. The Menger Sponge is a three-dimensional fractal shape made of cubes.
	The construction is done recursively and the total number of cubes is 20^d, where d represents the depth of recursion. 
 Usage:
	In After Effects CS4 or later, run the script
	Specify settings  
	Click on Create to create the Menger Sponge (a comp called Menger Comp is created)
 */


// This class represents the main class of the script
function Create3DMengerSponge()
{
	// Variable used to keep track of 'this' reference
	var create3DMengerSponge = this;
	
	// Create an instance of the utils class to use its functions
	var utils = new Create3DMengerSpongeUtils();

	// Script infos
	this.scriptMinSupportVersion = "9.0";
	this.scriptName = "Create3DMengerSponge.jsx";	
	this.scriptVersion = "2.6";
	this.scriptTitle = "Create 3D Menger Sponge";
	this.scriptCopyright = "Copyright (c) 2010 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = {en:"This script creates a 3D Menger Sponge.\\r\\rThe Menger Sponge is a three-dimensional fractal shape made of cubes. The construction is done recursively and the total number of cubes is 20^d, where d represents the depth of recursion (the value of this parameter must be small).", 
							  fr:"Ce script crée un éponge de Menger.\\r\\rL\\'éponge de Menger est une forme fractale tridimensionnelle composée de cubes. La construction se fait de façon récursive et le nombre total de cubes est 20^d, où d représente la profondeur de la récursion (la valeur de ce paramètre doit donc être petite)."};
	this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), 
						fr:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};		
	this.scriptUsage = {en:	"\u25BA In After Effects CS4 or later, run the script \\r" +
							"\u25BA Specify Sponge settings \\r" +
							"\u25BA Click on Create to create the Menger Sponge (a comp called Menger Comp is created)",
						fr:	"\u25BA Dans After Effects CS4 ou supérieur, exécuter le script \\r" +
							"\u25BA Spécifier les paramètres de construction \\r" +
							"\u25BA Cliquer sur Créer pour créer l\\\'éponge de Menger (une comp nommée Comp Menger est créée)"};
	
	// Errors & Warnings
	this.requirementErr = {en:"This script requires After Effects CS4 or later.", fr:"Ce script nécessite After Effects CS4 ou supérieur."};
	this.mengerConstructionErr = {en:"An error occurred while building the Menger Sponge.", fr:"Une erreur s'est produite pendant la construction de l'éponge de Menger."};	
	this.lotOfCubesWng = {en:"This will create a final comp containing %s cubes.\rDo you want to continue ?", fr:"La comp finale contiendra %s cubes.\rSouhaitez-vous continuer ?"};
		
	// UI strings 
	this.aboutBtnName = "?";
	this.settingsPnlName = {en:"Settings", fr:"Paramètres"};
	this.depthStName = {en:"Depth Of Recursion:", fr:"Profondeur récursion:"};
	this.depthEtDflt = 1;
	this.sizeStName = {en:"Size:", fr:"Taille:"};
	this.sizeEtDflt = 900;
	this.addControllerCbName = {en:"Add Controller And Light", fr:"Ajouter contrôleur et lumière"};
	this.runBtnName = {en:"Create", fr:"Créer"};
	
	// Miscellaneous
	this.cubeFolderName = {en:"Cube", fr:"Cube"};
	this.cubePrecompName = {en:"Cube Precomp", fr:"Précomp Cube"};
	this.cubeLayerNames = {en:["Front","Back","Left","Right","Bottom","Top"], fr:["Avant","Arrière","Gauche","Droite","Bas","Haut"]};
	this.controllerLayerName = {en:"Controller", fr:"Contrôleur"};
	this.buildCubeInfo = {en:"Build Cube ", fr:"Construit Cube "};
	this.completeInfo = {en:"Complete !", fr:"Terminé !"};
	
	this.finalCompName = {en:"Menger Comp", fr:"Comp Menger"};  
	this.finalCompWidth = 800;
	this.finalCompHeight = 600;  
	this.finalCompDuration = 16; // seconds 
	this.finalCompFramerate = 25;
	this.cubeColor = [1,1,1]; // white
	
	// Internal data
	this.depthOfRecursion = this.depthEtDflt;
	this.size = this.sizeEtDflt;
	this.cubesFolder = null;
	this.cubeSize = 0;
	this.totalCubes = 0;
	this.num = 0;
	this.positionMatrix = new Array();
	this.finalComp = null;			
	
	// Creates and displays the script interface
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});

		// resource specifications
		var res =
		"group { orientation:'column', alignment:['left','top'], alignChildren:['right','top'], \
			gr1: Group { \
				aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20] } \
			}, \
			gr2: Panel { text:'" + utils.loc(this.settingsPnlName) + "', alignment:['fill','top'], alignChildren:['right','top'], \
				gr21: Group { \
					depthSt: StaticText { text:'" + utils.loc(this.depthStName) + "' }, \
					depthEt: EditText { text:'" + this.depthEtDflt + "', characters:5 } \
				}, \
				gr22: Group { \
					sizeSt: StaticText { text:'" + utils.loc(this.sizeStName) + "' }, \
					sizeEt: EditText { text:'" + this.sizeEtDflt + "', characters:5 } \
				} \
			}, \
			gr3: Group { alignment:['fill','top'], \
				addControllerCb: Checkbox { text:'" + utils.loc(this.addControllerCbName) + "', value:true, alignment:['left','center'] }, \
				runBtn: Button { text:'" + utils.loc(this.runBtnName) + "', alignment:['right','center'] } \
			} \
		}"; 
		pal.gr = pal.add(res);
		
		// event callbacks
		pal.gr.gr1.aboutBtn.onClick = function () 
		{ 
			utils.createAboutDlg(create3DMengerSponge.scriptAbout, create3DMengerSponge.scriptUsage); 
		};
		
		pal.gr.gr2.gr21.depthEt.onChange = function () 
		{
			if (isNaN(this.text) || parseInt(this.text) <= 0) this.text = create3DMengerSponge.depthEtDflt;
			else if (parseInt(this.text) >= 3)
			{
				create3DMengerSponge.totalCubes = Math.pow(20, parseInt(this.text));
				
				if (!confirm(utils.loc(create3DMengerSponge.lotOfCubesWng).replace('%s', create3DMengerSponge.totalCubes))) 
				{
					this.text = create3DMengerSponge.depthEtDflt;				
				}
			}
			else this.text = parseInt(this.text);
		};
		
		pal.gr.gr2.gr22.sizeEt.onChange = function ()
		{
			if (isNaN(this.text) || parseInt(this.text) < 100) this.text = create3DMengerSponge.sizeEtDflt;
			else this.text = Math.round(this.text);
		};		   

		pal.gr.gr3.runBtn.onClick = function () 
		{ 
			create3DMengerSponge.createMengerSponge(pal); 
		};
		
		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}	   
	};

	// Creates and positions a 3D solid
	this.createSolid = function (comp, color, name, width, height, par, pos, ori)
	{
		var layer = comp.layers.addSolid(color, name, width, height, par);
		layer.threeDLayer = true;
		layer.position.setValue(pos);
		layer.orientation.setValue(ori);
	};   

	// Creates and positions a cube
	this.createCube = function (centerPos, size)
	{   
		size = Math.floor(size);
		var half = size / 2;
		
		/* 
		var resize = false;
		var rescale;		
		if (size < 4) 
		{
			rescale = (100 * size) / 4;
			resize = true;
			size = 4;
		}
		*/	   
		
		// Info palette
		clearOutput();
		writeLn(utils.loc(this.buildCubeInfo) + this.num + "/" + this.totalCubes);
		
		// Create cube precomp
		this.num ++;		
		var preComp;		
		if (this.num == 1) // first time we arrive here
		{
			preComp = app.project.items.addComp(utils.loc(this.cubePrecompName), size, size, 1.0, this.finalCompDuration, this.finalCompFramerate);
			preComp.parentFolder = this.cubeFolder;
		
			// Build cube
			var pos = [[half, half, 0], [half, half, size], [0, half, half], [size, half, half], [half, size, half], [half, 0, half]];
			var ori = [[0,0,0], [0,180,0], [0,90,0], [0,270,0], [90,0,0], [270,0,0]];
			for (var i = 0; i < 6; i++)
			{
				this.createSolid(preComp, this.cubeColor, utils.loc(this.cubeLayerNames)[i], size, size, 1.0, pos[i], ori[i]);			
			}			
			this.cubePrecomp = preComp;
		}
		else // the cube precomp already exists, so we just reuse it
		{
			preComp = this.cubePrecomp;
		}
				
		var precompLayer = this.finalComp.layers.add(preComp);
		precompLayer.threeDLayer = true;
		precompLayer.collapseTransformation = true;
		
		precompLayer.position.setValue(centerPos);	  
		precompLayer.anchorPoint.setValue([half,half,half]);
		
		/*
		if (resize) precompLayer.scale.setValue([rescale,rescale,rescale]);
		*/ 
		
		// Rename current cube
		var numStr = "" + this.num;
		var leadingZero = 3;
		while (numStr.length < leadingZero) numStr = "0" + numStr;
		precompLayer.name = utils.loc(this.cubePrecompName) + " - " + numStr;
	};

	// Fill the current level with cubes
	this.cubes = function (x, y, z, s, d)
	{
		var third = s / 3;
		for (var i = -third; i <= third; i += third)
		for (var j = -third; j <= third; j += third)
		for (var k = -third; k <= third; k += third)
		{
			if (Math.abs(i) + Math.abs(j) + Math.abs(k) >= 2 * third)
			{
				this.MengerSponge(x+i, y+j, z+k, third, d);
			}
		}
	};
	
	// Functional part of the script: recursively creates cubes
	this.MengerSponge = function (x, y, z, s, d)
	{
		if (d > 0)
		{
			d--;
			this.cubes(x, y, z, s, d);
		}
		else
		{
			this.positionMatrix.push([x, y, z]);
			this.createCube([x, y, z], s);
			this.positionMatrix.pop();
		}
	};

	// Add a 3D null to control the Menger Sponge
	this.addController = function (comp)
	{
		// add default light to see something more attractive
		comp.layers.addLight("Light", [comp.width/2, comp.height/2]);
		
		// add controller
		var controllerLayer = comp.layers.addNull();
		controllerLayer.threeDLayer = true;
		controllerLayer.name = utils.loc(this.controllerLayerName);
		
		// move layer to the center of the sponge
		controllerLayer.position.setValue([0,0,0]);
		
		// parenting (skip the first two layers - null and light layers)
		for (var i = 3; i <= comp.numLayers; i++) comp.layer(i).parent = controllerLayer;
			
		// reposition 
		controllerLayer.position.setValue([comp.width / 2, comp.height / 2, (Math.pow(3, this.depthOfRecursion) * this.cubeSize) / 2]);
		
		// little animation for fun...
		/*
		var keyTimes = [0, this.finalCompDuration];
		var keyValuesRotX = [-25, -25 + 2 * 360];
		var keyValuesRotY = [90, 90 + 2 * 360];
		controllerLayer.rotationX.setValuesAtTimes(keyTimes, keyValuesRotX);
		controllerLayer.rotationY.setValuesAtTimes(keyTimes, keyValuesRotY);
		*/
		controllerLayer.selected = true;
	};
	
	// Initializes data and calls the main routine
	this.createMengerSponge = function (pal)
	{
		try
		{
			var proj = app.project;
			
			app.beginUndoGroup(this.scriptTitle);
						
			// Create final comp
			this.finalComp = proj.items.addComp(utils.loc(this.finalCompName), this.finalCompWidth, this.finalCompHeight, 1.0, this.finalCompDuration, this.finalCompFramerate);
										   
			// Create folder to store cube precomp
			this.cubeFolder = proj.items.addFolder(utils.loc(this.cubeFolderName));
			
			// Retrieve settings from UI
			this.depthOfRecursion = parseInt(pal.gr.gr2.gr21.depthEt.text);
			this.size = parseInt(pal.gr.gr2.gr22.sizeEt.text);
			
			// Initialization
			this.positionMatrix = new Array();
			this.positionMatrix.push([0, 0, 0]);
			this.totalCubes = Math.pow(20, this.depthOfRecursion);
			this.cubeSize = this.size / (Math.pow(3, this.depthOfRecursion));				
			this.num = 0; 
					
			// Build Menger Sponge
			var err = this.mengerConstructionErr;
			try
			{
				this.MengerSponge(0, 0, 0, this.size, this.depthOfRecursion);
			}
			catch(e)
			{
				throw(err);
			};
			
			// Add controller
			if (pal.gr.gr3.addControllerCb.value)
			{
				this.addController(this.finalComp);
			}
			
			// Info palette
			clearOutput();
			writeLn(utils.loc(this.completeInfo));
				
			app.endUndoGroup();
		}
		catch(err)
		{
			utils.throwErr(err);
		}				
	};
	
	// Runs the script  
	this.run = function (thisObj) 
	{
		if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
		{
			utils.throwErr(this.requirementErr);
		}
		else
		{
			this.buildUI(thisObj);
		}	
	};
}


// This class provides some utility functions
function Create3DMengerSpongeUtils()
{
	// Variable used to keep track of 'this' reference
	var utils = this;
	
	// String localization function: english and french languages are supported
	this.loc = function (str)
	{
		return app.language == Language.FRENCH ? str.fr : str.en;
	};

	// Displays a window containg a localized error message
	this.throwErr = function (err)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
	};			

	// Displays a window containg a localized informative message
	this.throwMsg = function (msg)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script info:\r" + this.loc(msg), wndTitle, true);
	};
	
	// Displays a dialog containg the script description and usage
	this.createAboutDlg = function (aboutStr, usageStr)
	{	
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%63%61%74%65%67%6F%72%79%2F%73%63%72%69%70%74%73%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));		
	};
}


// Creates an instance of the main class and run it
new Create3DMengerSponge().run(this);
